/*

     Simple Freebus Sniffer application
            
 Copyright (C) Dirk Tostmann, 2009.
              
  tostmann [at] busware [dot] de
      www.busware.de
*/

/*
  Copyright 2009  Dirk Tostmann (tostmann [at] busware . de)

  Permission to use, copy, modify, and distribute this software
  and its documentation for any purpose and without fee is hereby
  granted, provided that the above copyright notice appear in all
  copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include "Descriptors.h"

#include "../../lufa/serial.h"

#include <util/parity.h>

TASK(Worker_Task);

/* Scheduler Task List */
TASK_LIST
{
	{ Task: USB_USBTask          , TaskStatus: TASK_STOP },
	{ Task: CDC_Task             , TaskStatus: TASK_STOP },
};

/** Main program entry point. This routine configures the hardware required by the application, then
 *  starts the scheduler to run the application tasks.
 */

volatile uint8_t xx   = 0;
volatile uint8_t data = 0;
volatile uint8_t bit  = 0;
volatile uint8_t bytes[ 20 ];
volatile uint8_t bc   = 0;
volatile uint8_t crc  = 0;

// Start bit detection!
ISR(INT2_vect) {

     // sync
     TCNT1  = -17;
     xx = 0;

     TIFR1 |= _BV( OCF1A ); // bittime 
     TIMSK1 = _BV( OCIE1A );

     bit  = 0;
     data = 0;
     
     PORTC  ^= _BV(PC4);		
//     printf_P( PSTR("S") );

     EIMSK &= ~_BV( INT2 ); // off!
}

// check bit
ISR(TIMER1_COMPA_vect) {

     uint8_t B = bit_is_set(PIND, PD2) ? 0 : 1;

     uint8_t err = 0;
     
     switch (bit) {
     case 0:
     case 1:
     case 2:
     case 3:
     case 4:
     case 5:
     case 6:
     case 7:
	  data |= (B<<bit);
	  break;
	  
     case 8: 
	  // Parity
	  if (parity_even_bit( data ) != B) {
	       err = 100;
	  }
	  break;

     case 9: 
	  // Stopbit
	  if (!B) {
	       err = 101;
	  }
	  break;

     case 10: 
	  // Pause
	  if (!B) {
	       err = 102;
	  }

	  bytes[bc++] = data;
	  crc = crc ^ data;
	  
	  break;

     case 11:
	  // Pause 
	  // reenable!
	  EIFR  |= _BV(INTF2);
	  EIMSK |= _BV(INT2);
	  
	  break;
     
     case 12:
	  // end of datagram

	  TIMSK1 = 0;

	  if ((bc > 1) && (crc != 0xff)) { // might be an ACK/NACK
	       err = 110;
	       
	  } else {
	       
	       for (uint8_t i=0;i<bc;i++)
		    printf_P( PSTR("%02X "), bytes[i] );
	       printf_P( PSTR("OK\r\n") );

	  }
	  
	  bc  = 0;
	  crc = 0;
	  break;

     }
     
     bit++;

     if (err) {
	  // Error - should wait for pause .. ?!
	  printf_P( PSTR("ERR %d\r\n"), err );
	  
	  TIMSK1 = 0;
	  bc  = 0;
	  crc = 0;

	  EIFR  |= _BV(INTF2);
	  EIMSK |= _BV(INT2);
     }
     
     TIFR1 |= _BV( OCF1A );
}

int main(void) {
     /* Disable watchdog if enabled by bootloader/fuses */
     MCUSR &= ~(1 << WDRF);
     wdt_disable();
     
     /* Disable Clock Division */
     SetSystemClockPrescaler(0);

     // LED as output
     DDRC  |= _BV(PC4);	
     
     // RX Interrupt
     EICRA |= _BV( ISC21 ) | _BV( ISC20 ); // rising edge
     EIMSK |= _BV( INT2 );

     // Timer 1
     TCNT1  = 0;
     TCCR1A = 0;
     TCCR1B = _BV(WGM12) | _BV(CS11);   // prescale 8 - 1usec - CTC
     OCR1A  = 103;         // BitZeit
     OCR1B  = 34;          // LowZeit
     OCR1C  = 70;          // CollTest

     /* Initialize Scheduler so that it can be used */
     Scheduler_Init();
     
     /* USB Init */
     LUFA_Init();

     sei();

     /* Scheduling - routine never returns, so put this last in the main function */
     Scheduler_Start();
}


